/*******************************************************************************
* Nine Men's Morris                                                            *
********************************************************************************
* (C) 2005-2011 Dr. Klaus Rieger                                               *
********************************************************************************
* $Rev:: 3                                                  $: Revision        *
* $Author:: rieger                                          $: Author          *
* $Date:: 2010-01-12                                        $: Date            *
*******************************************************************************/


package nmm;


import java.awt.*;						// creating user interfaces; painting graphics and images
import java.awt.event.*;				// for KeyListener (essential member: keyPressed, keyReleased, keyTyped)
import java.io.*;						// system input and output through data streams e.g. to open a file
import java.lang.Integer;				// for Integer.parseInt(string)

public class NMM extends Frame implements KeyListener
{
	private static final String PACKAGE_NAME	= "Nine Men's Morris";
	private static final String COPYRIGHT		= "(C) 2005-2011 Dr. Klaus Rieger";
	private static final String REVISION		= "$Rev: 3 $";
	private static final long serialVersionUID	= Long.parseLong(REVISION.substring(6, REVISION.length() - 2));

	static int EXTRAX	= 0;			// enlarge window in case of mintaka
	static int EXTRAY	= 0;

	static final int SCREENX = 306;		// size of client area
	static final int SCREENY = 234;		//

	static final int SLEEP = 20;		// time between stars
	static final int STEPS = 49;		// steps of stars
	static int step = -1;				// -1 manual mode, no star
										// 0...23 possible automatic moves starting on the field
										// 24, 34 possible automatic moves starting from the stack
	static int redrawfull = 0;			// full redraw

	static String PRESET = "";			// "" no preset string, "[48 characters]" preset string
	static int [] PLAYER = {99, 4};		// {white, black}, 99 manual, 0, ..., 9 automatic, -1, ..., -9 automatic with GRID
	static int MOVES = 100;				// maximum number of moves before game is over
	static int move = 0;				// if move == MOVES => modus = 0 (draw)
	static int skip = 0;				// 1 if last move was skipped, else 0

	static final long GRID_WAIT = 512;	// maximum time [s] for first job, recommended: 16, 32, 64, 128, 256, 512, ...
	static long grid_wait = 0;			// (second job +GRID_WAIT/2, third job +GRID_WAIT/4, ..., 1, 1, 1)
	static int grid_status = 0;			// 0 idle, 1 send jobs to GRID, 2 receive jobs from GRID
	static int grid_jobnumber = 0;		// every job gets unique number (000000, 000001, ...)
	static int grid_movestart = 0;		// job number of the first job in one move

	static int HOSTS = 1;				// number of hosts for GRID
	static String hosts[] = {"mintaka.ari.uni-heidelberg.de", "", "", "", "", "", "", "", "", "",
							 "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
							 "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",};
	static int [] host_wait =			// waiting time [s] of a host
	{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
	static int [] host_timeout = 		// number of timeouts of a host
	{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
	static int [] move_timeout = 		// number of timeouts of a host during current move
	{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
	static long [] duration = {0, 0};	// {white, black}, total time [ms]
	static long time;					// time [ms] since January 1, 1970 UTC at begin of move

	static int terminate = 0;			// if 2 < terminate exit
	static int modus = 1;				// 1 run, 0 draw, -1 white lost, -2 black lost
	static int caged = 0;				// 1 lost last move because no move possible (for undo)

	static int ATTACK[] = {5, 5};		// aggressiveness !!! wrong in case of different value and GRID !!!
	static int best_from;				// best move from
	static int best_to;					// best move to
	static int best_capture;			// best capture
	static int best_scope;				// lowest scope of rival = most constrained movement of rival
										//  10000: nothing found, lost
										//   9x00: sth. found, will loose
										//  -9x00: sth. found, will win	
										// -10000: sth. found, win
	static Token token = new Token(ATTACK[0]);// for class Token see Token.java


	public static void main(String[] args)
	{
		int player = 0;								// 0, 2, 4, ... white, 1, 3, 5, ...black
		String auto[] = {"a0", "a1", "a2", "a3", "a4", "a5", "a6", "a7", "a8", "a9"};
		String grid[] = {"g0", "g1", "g2", "g3", "g4", "g5", "g6", "g7", "g8", "g9"};
		String help[] = {"-h", "-help", "--h", "--help"};

		for (int a=0; a<args.length; a++)			// parse command line arguments
		{
			for (int b=0; b<4; b++)
			{
				if (args[a].equals(help[b]))
				{
					System.out.print(PACKAGE_NAME); System.out.print(", Version "); System.out.print(serialVersionUID);
					System.out.print(", Copyright "); System.out.println(COPYRIGHT);
					System.out.println("Usage: java -jar NMM.jar [moves] [white] [black] [preset]");
					System.out.println("Abstract strategy board game for two players named 'Nine Men's Morris'.");
					System.out.println("");
					System.out.println("Options:");
					System.out.println("  -h, -help            Shows Help");
					System.out.println("  --h, --help          Shows Help");
					System.out.println("");
					System.out.println("  Maximum number of moves before game is over:");
					System.out.println("  (Negative numbers combined with preset string result in calculation without GUI.)");
					System.out.println("  1                    One move (doesn't make sense)");
					System.out.println("  2                    Two moves (doesn't make sense)");
					System.out.println("  ...                  ... moves (default is 100)");
					System.out.println("");
					System.out.println("  First occurrence white, second occurrence black:");
					System.out.println("");
					System.out.println("  m                    Manual player");
					System.out.println("");
					System.out.println("  a0                   Calculation depth 0: very fast, very stupid");
					System.out.println("  ...");
					System.out.println("  a9                   Calculation depth 9: very slow, very intelligent");
					System.out.println("");
					System.out.println("  g0                   Calculation depth 0, like a0");
					System.out.println("  g1                   Calculation depth 1, using the GRID");
					System.out.println("  ...");
					System.out.println("  g9                   Calculation depth 9, using the GRID");
					System.out.println("");
					System.out.println("  The preset string consists of precisely 48 characters");
					System.out.println("  - first 44 characters describe a position (0 if empty, 1 if white, 2 if black)");
					System.out.println("  - following 4 characters are the digits of the actual move");
					System.out.println("");
					System.out.println("");
					System.out.println("Use:                   Click the title bar to change background color.");
					System.out.println("                       - Clicking three times within one move may end the program.");
					System.out.println("                       - Alternatively you can use your keyboard pushing 'Esc'.");
					System.out.println("                       Moving and capturing tokens:");
					System.out.println("                       - If it is your turn, first click the token you want to move,");
					System.out.println("                         then the place you want to move the token to.");
					System.out.println("                       - If it is your turn to capture a token,");
					System.out.println("                         just click the token of your choice.");
					System.out.println("                       Click the footer to undo the last move.");
					System.out.println("                       - Alternatively you can use your keyboard pushing 'Backspace'.");
					System.out.println("");
					System.out.println("");
					System.out.println("Example 1:             java -jar NMM.jar 50 m a1");
					System.out.println("Result:                Maximum 50 moves before game is over.");
					System.out.println("                       Human plays white, computer plays black fast and stupidly.");
					System.out.println("");
					System.out.println("");
					System.out.println("Example 2:             java -jar NMM.jar g8 m");
					System.out.println("Result:                Maximum 50 moves before game is over.");
					System.out.println("                       Computer uses GRID to play white slowly and intelligently, human plays black.");
					System.out.println("");
					System.out.print(PACKAGE_NAME); System.out.print(", "); System.out.println(serialVersionUID);
					System.exit(0);
				}
			}
			if (args[a].length() == 48)				// preset string
			{
				PRESET = args[a];
			}
			else
			{
				try									// argument is a number
				{
					MOVES = Integer.parseInt(args[a]);
				}
				catch (NumberFormatException e)		// argument is not a number
				{
					if (args[a].equals("m"))
					{
						PLAYER[player] = 99;
						player = 1 - player;
					}
					else
					{
						for (int b=0; b<10; b++)
						{
							if (args[a].equals(auto[b]))
							{
								PLAYER[player] = b;
								player = 1 - player;
							}
							if (args[a].equals(grid[b]))
							{
								PLAYER[player] = -b;
								player = 1 - player;
							}
						}
					}
				}
			}
		}

		if (MOVES == 0)
		{
			System.out.println("Error parsing command line: moves == 0 illegal");
			System.out.println("For more info use 'java -jar NMM.jar --help'");
			System.exit(0);
		}

		if (MOVES < 0)
		{
			if (PRESET == "")
			{
				System.out.println("Error parsing command line: moves < 0 only allowed with preset string");
				System.out.println("For more info use 'java -jar NMM.jar --help'");
				System.exit(0);
			}
			else									// moves < 0 and PRESET
			{
				if (((PLAYER[0] < 0) || (PLAYER[0] == 99)) || ((PLAYER[1] < 0) || (PLAYER[1] == 99)))
				{
					System.out.println("Error parsing command line: moves < 0 only allowed with white and black a0...a9");
					System.out.println("For more info use 'java -jar NMM.jar --help'");
					System.exit(0);
				}
				else								// PLAYER[0/1] = 0...9
				{
					MOVES = -MOVES;					// NMM without GUI
					move = token.string2location(PRESET);
					System.out.println(token.calculate(PLAYER[move%2], 0));
					System.exit(0);
				}
			}
		}
		else
		{
			String str = "";
			try
			{
				Process p = Runtime.getRuntime().exec("hostname");
				BufferedReader in = new BufferedReader(new InputStreamReader(p.getInputStream()));
				while ((str = in.readLine()) != null)
				{
					if (str.equals("mintaka"))
					{
						EXTRAX = 22;
						EXTRAY = 31;
					}
				}
		    }
			catch (IOException e) {e.printStackTrace();}
			
			try
			{
				BufferedReader in = new BufferedReader(new FileReader("host.txt"));
				if ((str = in.readLine()) != null)
				{
					HOSTS = 0;
					do
					{
						hosts[HOSTS++] = str; 
					} while ((str = in.readLine()) != null);
				}
			} catch (IOException e){}

			NMM wnd = new NMM(PRESET);				// NMM with GUI
			wnd.setSize(wnd.getInsets().left + SCREENX + wnd.getInsets().right + EXTRAX,
						wnd.getInsets().top + SCREENY + wnd.getInsets().bottom + EXTRAY);

			System.out.println("");
			System.out.println("Nine Men's Morris");
			System.out.println("*****************");
			for (int a=0; a<2; a++)
			{
				if (a == 0) System.out.print("White:  ");
					   else System.out.print("Black:  ");
				if (PLAYER[a] == 99) System.out.println("Manual");
				else
				{				
					System.out.println("Automatic");
					if (PLAYER[a] < 0)
					{
						System.out.print("        Depth = ");
						System.out.println(-PLAYER[a]);
						System.out.print("        GRID  = ");
						System.out.println(-PLAYER[a] - 1);
					}
					else
					{
						System.out.print("        Depth = ");
						System.out.println(PLAYER[a]);
					}
				}
			}
			System.out.print("Moves:  "); System.out.println(MOVES);
			System.out.print("Preset: "); if (PRESET == "") System.out.println("No"); else System.out.println(PRESET);
			System.out.print("Screen: (x+"); System.out.print(EXTRAX); System.out.print(",y+"); System.out.print(EXTRAY);System.out.println(")");
			System.out.print("Hosts:  "); System.out.println(HOSTS);
			for (int a=0; a<HOSTS; a++) System.out.println(hosts[a]);
			System.out.println("*****************");
			System.out.println("");
			time = System.currentTimeMillis();		// start time for the first move

			do
			{
				try
				{
					Thread.sleep(SLEEP);
				} catch (InterruptedException e) {e.printStackTrace();}
				wnd.repaint();
	
				if (step == -1)
				{									// start automatic movements
					if ((PLAYER[move%2] != 99) && (modus == 1)) step = 0;
				}
			} while (terminate < 3);
			System.exit(0);
		}
	}

	public NMM(final String str)
	{
		super("NMM  ©2011 Klaus Rieger");
		move = token.string2location(str);
		addWindowListener(new NMMClosing(true));
		addKeyListener(this);
		addMouseListener(new MouseAdapter()
		{
			public void mousePressed(MouseEvent e)	// invoked when a mouse button has been pressed on a component
			{
				int x, y, id, options;

				x = (e.getX() - getInsets().left - 6)/6;
				y = (e.getY() - getInsets().top - 20)/10;
				if (y == -1)						// click of title bar
				{
					terminate++;
					redrawfull = 1;
				}
				else
				{
					if (19 < y)						// click on the footer
					{
						if (skip == 1) move--;
						switch (token.undo())
						{
							case 5: System.out.println("undo marking and moving");
									if ((move-- == MOVES) && (modus == 0)) modus = 1;		// one move back
									break;
							case 4: System.out.println("undo last move");
									if ((move-- == MOVES) && (modus == 0)) modus = 1;		// one move back
									break;
							case 3: System.out.println("undo move");
									break;
							case 2: System.out.println("undo marking");
									break;
							case 1: System.out.println("undo not possible");
									break;
							case -1:
							case -2: modus = 1;		// not lost
									 System.out.print("undo loosing - ");
							case  0: System.out.println("undo capture");
									 if ((move-- == MOVES) && (modus == 0)) modus = 1;		// one move back
						}
						step = -1;
						redrawfull = 1;
						terminate = 0;
						skip = 0;
						if ((modus < 0) && (caged == 1)) modus = 1; 						// game goes on
						caged = 0;
					}
					else
					{
						id = token.search(x, y);
						options = -3;
						if (-1 < id)
						{
							if ((token.read(id) < 0) || (token.read(id) == 1 + move%2)) options = token.marking(id);
						}	// if marking or own color calculate options
						System.out.print("id: ");
						System.out.print(id);
						System.out.print(" - options: ");
						System.out.println(options);
						if ((options == -1) || (options == -2))
						{							// game over
							if (modus == 1) modus = options;
							if (options == -1) System.out.println("Game Over - White lost!");
										  else System.out.println("Game Over - Black lost!");
							duration[move%2] += System.currentTimeMillis() - time;
							time = System.currentTimeMillis();
							move++;
							System.out.println(token.location2string(move));
							System.out.println("");
							redrawfull = 1;
							terminate = 0;
							skip = 0;
						}
						if (options == 0)			// move completed
						{
							skip = 0;
							caged = 0;
							duration[move%2] += System.currentTimeMillis() - time;
							time = System.currentTimeMillis();
							if ((++move == MOVES) && (modus == 1))
							{
								modus = 0;
								System.out.println("Game Over - Draw!");
							}
							if (token.score(1 + move%2) == 0)
							{
								if (modus == 1)		// next can't move => lost
								{
									modus = -1 - move%2;
									if (modus == -1) System.out.println("Game Over - White lost!");
									if (modus == -2) System.out.println("Game Over - Black lost!");
									caged = 1;
								}
								duration[move%2] += System.currentTimeMillis() - time;
								time = System.currentTimeMillis();
								move++;				// skip other color
								skip = 1;
							}
							System.out.println(token.location2string(move));
							System.out.println("");
							redrawfull = 1;
							terminate = 0;
						}
					}
				}
			}

			public void mouseReleased(MouseEvent e)	// mouse button has been released on a component
			{
			}

			public void mouseEntered(MouseEvent e)	// mouse enters a component
			{
			}

			public void mouseExited(MouseEvent e)	// mouse exits a component
			{
			}

			public void mouseClicked(MouseEvent e)	// mouse button has been clicked (pressed and released) on a component
			{
			}
		});
		setBackground(Color.blue);					// set window color
		setVisible(true);							// set window visible
	}


	public void keyTyped(KeyEvent e)	// key typed event from the text field
	{
		if (e.getKeyChar() == KeyEvent.VK_ESCAPE)
		{
			terminate++;
			redrawfull = 1;
		}
		if (e.getKeyChar() == KeyEvent.VK_BACK_SPACE)
		{
			if (skip == 1) move--;
			switch (token.undo())
			{
				case 5: System.out.println("undo marking and moving");
						if ((move-- == MOVES) && (modus == 0)) modus = 1;		// one move back
						break;
				case 4: System.out.println("undo last move");
						if ((move-- == MOVES) && (modus == 0)) modus = 1;		// one move back
						break;
				case 3: System.out.println("undo move");
						break;
				case 2: System.out.println("undo marking");
						break;
				case 1: System.out.println("undo not possible");
						break;
				case -1:
				case -2: modus = 1;	// not lost
						 System.out.println("undo loosing - ");
				case  0: System.out.println("undo capture");
						 if ((move-- == MOVES) && (modus == 0)) modus = 1;		// one move back
			}
			step = -1;
			redrawfull = 1;
			terminate = 0;
			skip = 0;
			if ((modus < 0) && (caged == 1)) modus = 1; 						// game goes on
			caged = 0;
		}
	}


	public void keyPressed(KeyEvent e)	// key-pressed event from the text field
	{
	}


	public void keyReleased(KeyEvent e)	// key-released event from the text field
	{
	}


	public void paint(Graphics g)
	{
		Font font = new Font("Lucida Sans Typewriter Regular", Font.PLAIN, 10);
		g.setFont(font);														// set font

		setSize(getInsets().left + SCREENX + getInsets().right + EXTRAX,		// set window size
				getInsets().top + SCREENY + getInsets().bottom + EXTRAY);

		String [] field = {	"         ---------------   ---------------         ","       |                 |                 |       ",
							"       |                 |                 |       ","       |       ---------   ---------       |       ",
							"       |     |           |           |     |       ","       |     |           |           |     |       ",
							"       |     |       ---   ---       |     |       ","       |     |     |           |     |     |       ",
							"       |     |     |           |     |     |       ","         ---   ---               ---   ---         ",
							"       |     |     |           |     |     |       ","       |     |     |           |     |     |       ",
							"       |     |       ---   ---       |     |       ","       |     |           |           |     |       ",
							"       |     |           |           |     |       ","       |       ---------   ---------       |       ",
							"       |                 |                 |       ","       |                 |                 |       ",
							"         ---------------   ---------------         "};

		switch (terminate)
		{
			case 0: g.setColor(Color.blue);
					break;
			case 1: g.setColor(Color.yellow);
					break;
			case 2: g.setColor(Color.black);
					break;
			default:g.setColor(Color.white);
		}
		g.fillRect(getInsets().left, getInsets().top, SCREENX, SCREENY);

		String headline = "";
		String extracharacter;
		String footer = "";

		if (10000 <= move + 1) extracharacter = "";
		else
		{
			if (1000 <= move + 1) extracharacter = " ";
			else
			{
				if (100 <= move + 1) extracharacter = "  ";
				else
				{
					if (10 <= move +1) extracharacter = "   ";
					              else extracharacter = "    ";
				}
			}
		}

		if (modus == 1)
		{
			if (0 == (move%2)) headline = "Move: " + (move + 1) + extracharacter + "                         Move by White";
						  else headline = "Move: " + (move + 1) + extracharacter + "                         Move by Black";
		}
		else
		{
			if (modus ==  0) headline = "Move: " + (move + 1) + extracharacter + "                      Game was a Draw!";
			if (modus == -1) headline = "Move: " + (move + 1) + extracharacter + "                           Black Wins!";
			if (modus == -2) headline = "Move: " + (move + 1) + extracharacter + "                           White Wins!";
		}

		for (int a=0; a<STEPS; a++)
			if (a < step) footer += "#";
					 else footer += " ";

		g.setColor(Color.white);
		g.drawString(" _________________________________________________", getInsets().left, getInsets().top + 1*10 + 3);

		for (int a=0; a<19; a++)
			g.drawString(field[a], getInsets().left, getInsets().top + (3 + a)*10);

		g.drawString(" _________________________________________________", getInsets().left, getInsets().top + 22*10 - 2);

		g.setColor(Color.yellow);
		g.fillRect(getInsets().left, getInsets().top, SCREENX, 12);
		g.fillRect(getInsets().left, getInsets().top + 23*10 - 8, SCREENX, 12);
		g.setColor(Color.black);
		g.drawString(headline, getInsets().left + 6, getInsets().top + 1*10 - 1);
		g.drawString(footer, getInsets().left + 6, getInsets().top + 23*10 + 1);

		updateToken(g);
		redrawfull = 0;
	}


	public void update(Graphics g)
	{
		Font font = new Font("Lucida Sans Typewriter Regular", Font.PLAIN, 10);
		g.setFont(font);

		if (-1 < step)										// automatic movements
		{
			if (step == 0)									// prepare automatic movements
			{
				for (int a=0; a<24; a++)					// in case of manual undo of last captured ...
					if (token.read(a) < 0) token.undo();	// ... complete undo
				best_scope = 10000;							// scope of rival
				best_from = -1;								// best move from
				best_to = -1;								// best move to
				best_capture = -1;							// best capture

				System.out.print("move = ");
				System.out.println(move + 1);
				System.out.print("PLAYER[");
				System.out.print(move%2);
				System.out.print("] = ");
				System.out.println(PLAYER[move%2]);
				if (PLAYER[move%2] < 0)
				{
					if (grid_status == 0)					// first GRID round
					{
						grid_status = 1;					// send jobs to GRID
						grid_movestart = grid_jobnumber;
					}
					else									// second GRID round
					{
						grid_status = 2;
						grid_jobnumber = grid_movestart;	// receive jobs from GRID
						grid_wait = 0;
					}
					System.out.print("grid_status = ");
					System.out.println(grid_status);
					System.out.print("grid_movestart = ");
					System.out.println(grid_movestart);
				}
			}
															// 0...23 possible automatic moves starting on the field
			if (((step < 25) || (step == 34))				// 24, 34 possible automatic moves starting from the stack
			 && (-10000 < best_scope))						// -10000 win immediately => no further search
			{
				if (token.read(step) == 1 + move%2)			// if a token of the right color is on the search position
				{
					int options = token.marking(step);
					if (0 < options)
					{
						for (int a=0; a<24; a++)
						{
							if (token.read(a) < 0)
							{
								System.out.println("");
								System.out.print(step);
								System.out.print(">");
								System.out.print(a);
								options = token.marking(a);
								int scope;
								if (0 < options)			// mill
								{
									for (int b=0; b<24; b++)
									{
										if (token.read(b) < 0)
										{
											System.out.print(" mill\t");
											if (token.marking(b) < 0)		// -1 white has lost, -2 black has lost, -3 not possible
											{
												best_scope = -10000;
												best_from = step;
												best_to = a;
												best_capture = b;
												a = b = 24;					// no further testing
											}
											else							// -10000 < best_scope
											{
												if (grid_status == 1)		// send jobs to GRID
												{
													String exec = gridHost("globusrun-ws -submit -F HOST -s -so STDOUT -c JAVA -jar NMM.jar -MOVES DEPTH DEPTH STRING", grid_jobnumber);
													exec = exec.replaceAll("STDOUT", int2str6(grid_jobnumber));
													char[] chars = new char[4];
													chars[0] = (char)('0' + (MOVES/1000)%10);
													chars[1] = (char)('0' + (MOVES/100)%10);
													chars[2] = (char)('0' + (MOVES/10)%10);
													chars[3] = (char)('0' + MOVES%10);
													exec = exec.replaceAll("MOVES", new String(chars));
													exec = gridHost(exec, grid_jobnumber);
													switch(-PLAYER[move%2] - 1)
													{
														case 0: exec = exec.replaceAll("DEPTH", "a0");
																break;
														case 1: exec = exec.replaceAll("DEPTH", "a1");
																break;
														case 2: exec = exec.replaceAll("DEPTH", "a2");
																break;
														case 3: exec = exec.replaceAll("DEPTH", "a3");
																break;
														case 4: exec = exec.replaceAll("DEPTH", "a4");
																break;
														case 5: exec = exec.replaceAll("DEPTH", "a5");
																break;
														case 6: exec = exec.replaceAll("DEPTH", "a6");
																break;
														case 7: exec = exec.replaceAll("DEPTH", "a7");
																break;
														case 8: exec = exec.replaceAll("DEPTH", "a8");
																break;
														default: System.out.print("Depth error");
																 System.exit(0);
													}
													exec = exec.replaceAll("STRING", token.location2string(move));
													System.out.print(exec);
													try
													{
														Runtime.getRuntime().exec(exec);
													} catch (IOException e1) {e1.printStackTrace();}
													if (0 < host_timeout[++grid_jobnumber%HOSTS])
														if (1 < host_timeout[++grid_jobnumber%HOSTS])
															if (2 < host_timeout[++grid_jobnumber%HOSTS])
																if (3 < host_timeout[++grid_jobnumber%HOSTS]);
												}
												else
												{
													scope = 10000;
													if (grid_status == 0)	// no GRID
													{
														if (PLAYER[move%2] == 0)
														{
															scope = token.score_print(2 - move%2);
														}
														else
														{
															Token token_main = new Token(ATTACK[move%2]);
System.out.print("attack: ");
System.out.println(ATTACK[move%2]);
System.out.println(token_main.getAttack());
															token_main.string2location(token.location2string(move));
															scope = -token_main.calculate(PLAYER[move%2] - 1, 1);
														}
													}
													else					// receive jobs from GRID
													{
														String file = int2str6(grid_jobnumber);
														System.out.print(file);
														System.out.print(" - ");
														String str = null;
														grid_wait += GRID_WAIT >> (grid_jobnumber - grid_movestart);
														if (grid_wait == 0) grid_wait = 1;
														System.out.print(grid_wait);
														System.out.print("s - ");
														do
														{
															try
															{
																Thread.sleep(1000);
															} catch (InterruptedException e1) {System.out.print("Sleep error");}
															try
															{
																BufferedReader in = new BufferedReader(new FileReader(file));
																str = in.readLine();
																if (in != null) in.close();
															} catch (IOException e) {}
															try
															{
																scope = -Integer.parseInt(str);
															} catch (NumberFormatException e)
															{
																grid_wait--;
																host_wait[grid_jobnumber%HOSTS]++;
															};
														} while ((scope == 10000) && (0 < grid_wait));
														if (grid_wait == 0)
														{
															System.out.print("Timeout of ");
															System.out.print(gridHost("HOST", grid_jobnumber));
										 					System.out.print(" - ");
															move_timeout[grid_jobnumber%HOSTS]++;
															System.out.print("recalculate - ");
															Token token_main = new Token(ATTACK[move%2]);	// recalculate without GRID
System.out.print("attack: ");
System.out.println(ATTACK[move%2]);
System.out.println(token_main.getAttack());
															token_main.string2location(token.location2string(move));
															scope = -token_main.calculate(-PLAYER[move%2] - 1, 1);
															System.out.print("- recalculate: ");
														}
														System.out.print(scope);
														System.out.print(" ");
														if (0 < host_timeout[++grid_jobnumber%HOSTS])
															if (1 < host_timeout[++grid_jobnumber%HOSTS])
																if (2 < host_timeout[++grid_jobnumber%HOSTS])
																	if (3 < host_timeout[++grid_jobnumber%HOSTS]);
													}
													if (0 < scope)			// long term profit (token.value = -2 ... -4),
													{						// expecially if token on crossing is captured
														scope += token.value(b, 33) - ATTACK[move%2];	// -ATTACK more aggressive
														if (scope < 1) scope = 1;	// not lower than 1, because not best
													}
													else scope += token.value(b, 33) - ATTACK[move%2];	// -ATTACK more aggressive

													if (scope < best_scope)
													{
														best_scope = scope;
														best_from = step;
														best_to = a;
														best_capture = b;
													}

													if ((scope != 10000) && // if possible move found and ...
														(scope == best_scope))// ... in case ofequal result ...
													{						// ... take a look at position
														if (best_to == -1)	// previous choice without capture
														{					// => use new choice with capture
															best_scope = scope;
															best_from = step;
															best_to = a;
															best_capture = b;
														}
														else
														{					// prefer central mill
															if (token.value(best_from, best_to) < token.value(step, a))
															{
																best_scope = scope;
																best_from = step;
																best_to = a;
																best_capture = b;
															}				// prefer capturing central token
															if (token.value(best_from, best_to) == token.value(step, a))
															{
																if (0 < token.value(best_to, a))
																{
																	best_scope = scope;
																	best_from = step;
																	best_to = a;
																	best_capture = b;
																}
															}
														}
													}
												}
											}
											token.undo();
										}									// for (token.read(b) < 0) end
									}										// for (int b=0; b<24; b++) end
								}
								else										// no mill
								{
									System.out.print(" move\t");
									if (grid_status == 1)					// send jobs to GRID
									{
										String exec = gridHost("globusrun-ws -submit -F HOST -s -so STDOUT -c JAVA -jar NMM.jar -MOVES DEPTH DEPTH STRING", grid_jobnumber);
										exec = exec.replaceAll("STDOUT", int2str6(grid_jobnumber));
										char[] chars = new char[4];
										chars[0] = (char)('0' + (MOVES/1000)%10);
										chars[1] = (char)('0' + (MOVES/100)%10);
										chars[2] = (char)('0' + (MOVES/10)%10);
										chars[3] = (char)('0' + MOVES%10);
										exec = exec.replaceAll("MOVES", new String(chars));
										switch(-PLAYER[move%2] - 1)
										{
											case 0: exec = exec.replaceAll("DEPTH", "a0");
													break;
											case 1: exec = exec.replaceAll("DEPTH", "a1");
													break;
											case 2: exec = exec.replaceAll("DEPTH", "a2");
													break;
											case 3: exec = exec.replaceAll("DEPTH", "a3");
													break;
											case 4: exec = exec.replaceAll("DEPTH", "a4");
													break;
											case 5: exec = exec.replaceAll("DEPTH", "a5");
													break;
											case 6: exec = exec.replaceAll("DEPTH", "a6");
													break;
											case 7: exec = exec.replaceAll("DEPTH", "a7");
													break;
											case 8: exec = exec.replaceAll("DEPTH", "a8");
													break;
											default: System.out.print("Depth error");
													 System.exit(0);
										}
										exec = exec.replaceAll("STRING", token.location2string(move));
										System.out.print(exec);
										try
										{
											Runtime.getRuntime().exec(exec);
										} catch (IOException e1) {e1.printStackTrace();}
										if (0 < host_timeout[++grid_jobnumber%HOSTS])
											if (1 < host_timeout[++grid_jobnumber%HOSTS])
												if (2 < host_timeout[++grid_jobnumber%HOSTS])
													if (3 < host_timeout[++grid_jobnumber%HOSTS]);
									}
									else
									{
										scope = 10000;
										if (grid_status == 0)				// no GRID
										{
											if (PLAYER[move%2] == 0)
											{
												scope = token.score_print(2 - move%2);
											}
											else
											{
												Token token_main = new Token(ATTACK[move%2]);
System.out.print("attack: ");
System.out.println(ATTACK[move%2]);
System.out.println(token_main.getAttack());
												token_main.string2location(token.location2string(move));
												scope = -token_main.calculate(PLAYER[move%2] - 1, 1);
											}
										}
										else								// receive jobs from GRID
										{
											String file = int2str6(grid_jobnumber);
											System.out.print(file);
											System.out.print(" - ");
											String str = null;
											grid_wait += GRID_WAIT >> (grid_jobnumber - grid_movestart);
											if (grid_wait == 0) grid_wait = 1;
											System.out.print(grid_wait);
											System.out.print("s - ");
											do
											{
												try
												{
													Thread.sleep(1000);
												} catch (InterruptedException e1) {System.out.print("Sleep error");}
												try
												{
													BufferedReader in = new BufferedReader(new FileReader(file));
													str = in.readLine();
													if (in != null) in.close();
												} catch (IOException e) {}
												try
												{
													scope = -Integer.parseInt(str);
												} catch (NumberFormatException e)
												{
													grid_wait--;
													host_wait[grid_jobnumber%HOSTS]++;
												};
											} while ((scope == 10000) && (0 < grid_wait));
											if (grid_wait == 0)
											{
												System.out.print("Timeout of ");
												System.out.print(gridHost("HOST", grid_jobnumber));
												System.out.print(" - ");
												move_timeout[grid_jobnumber%HOSTS]++;
												System.out.print("recalculate - ");
												Token token_main = new Token(ATTACK[move%2]);	// recalculate without GRID
System.out.print("attack: ");
System.out.println(ATTACK[move%2]);
System.out.println(token_main.getAttack());
												token_main.string2location(token.location2string(move));
												scope = -token_main.calculate(-PLAYER[move%2] - 1, 1);
												System.out.print("- recalculate: ");
											}
											System.out.print(scope);
											System.out.print(" ");
											if (0 < host_timeout[++grid_jobnumber%HOSTS])
												if (1 < host_timeout[++grid_jobnumber%HOSTS])
													if (2 < host_timeout[++grid_jobnumber%HOSTS])
														if (3 < host_timeout[++grid_jobnumber%HOSTS]);
										}

										if (scope < best_scope)
										{
											best_scope = scope;
											best_from = step;
											best_to = a;
										}

										if ((scope != 10000) &&		// if possible move found and ...
											(scope == best_scope))	//... in case of equal result ...
										{							// ... take a look at position
											if (token.value(best_from, best_to) < token.value(step, a))
											{
												best_scope = scope;
												best_from = step;
												best_to = a;
												System.out.print("+!");
											}
										}
									}
								}
								token.undo();
								options = token.marking(step);
							}
						}
						token.undo();
					}
				}
			}
			else
			{
				if ((step < 25) || (step == 34))
				{
					System.out.println("");
					System.out.print(step);
					System.out.print("\t\tNo further calculation, because best_scope already ");
					System.out.print(best_scope);
				}
			}

			if (step++ == STEPS)					// end automatic movements
			{
				System.out.println("");
				if (grid_status == 1)				// grid_status == 1
				{									// => result not available
					step = 0;
					redrawfull = 1;
					System.out.println("");
				}
				else								// grid_status == 0 or grid_status == 2
				{									// => result available for interpretation
					if (grid_status == 2)
					{
						grid_status = 0;
						System.out.println("");
						System.out.println("*********************************************** wait time ***** timeout *****");
						for (int a=0; a<HOSTS; a++)
						{
							host_timeout[a] += move_timeout[a];
							System.out.print(gridHost("HOST", a));
							System.out.print("\t\t");
							if (gridHost("HOST", a).length() < 16) System.out.print("\t");
							if (gridHost("HOST", a).length() < 24) System.out.print("\t");
							if (gridHost("HOST", a).length() < 32) System.out.print("\t");
							System.out.print(host_wait[a]);
							System.out.print("s\t\t");
							System.out.print(host_timeout[a]);
							if (move_timeout[a] == 0) System.out.println("");
							else
							{
								System.out.print("  (+");
								System.out.print(move_timeout[a]);
								System.out.println(")");
								move_timeout[a] = 0;
							}
						}
						System.out.println("*****************************************************************************");
					}
					duration[move%2] += System.currentTimeMillis() - time;
					time = System.currentTimeMillis();
					move ++;
					System.out.print("Time white: ");
					System.out.print(duration[0]/1000);
					System.out.println("s");
					System.out.print("Time black: ");
					System.out.print(duration[1]/1000);
					System.out.println("s");
					if (best_from == -1)			// no move possible at all => lost
					{
						modus = move%2 - 2;
						if (modus == -1) System.out.println("Game Over - White lost!");
						if (modus == -2) System.out.println("Game Over - Black lost!");
					}
					else
					{
						System.out.print("best_from: ");
						System.out.println(best_from);
						System.out.print("best_to: ");
						System.out.println(best_to);
						token.marking(best_from);	// execute best move
						token.marking(best_to);
						if (-1 < best_capture)
						{							// game over: lost
							System.out.print("best_capture: ");
							System.out.println(best_capture);
							switch (token.marking(best_capture))
							{
								case -1: System.out.println("Game Over - White lost!");
										 modus = -1;
										 break;
								case -2: System.out.println("Game Over - Black lost!");
										 modus = -2;
										 break;
							}
						}
						System.out.print("best_scope: ");
						System.out.println(best_scope);
					}
					if ((modus == 1) && (move == MOVES))
					{								// game over: draw
						System.out.println("Game Over - Draw!");
						modus = 0;
					}
					skip = 0;
					caged = 0;
					if (token.score_print(1 + move%2) == 0)
					{
						System.out.println("");
						if (modus == 1)				// next can't move => lost
						{
							modus = -1 - move%2;
							if (modus == -1) System.out.println("Game Over - White lost!");
							if (modus == -2) System.out.println("Game Over - Black lost!");
							caged = 1;
						}
						duration[move%2] += System.currentTimeMillis() - time;
						time = System.currentTimeMillis();
						move++;						// skip other color
						skip = 1;
					}
					else System.out.println("");	// newline necessary in both cases but possible after if ... only
					step = -1;						// manual mode
					System.out.println(token.location2string(move));
					System.out.println("");
					redrawfull = 1;
					terminate = 0;
				}
			}
		}

		if (redrawfull == 1)
		{
			paint(g);
		}
		else
		{
			String footer = "";
			for (int a=0; a<STEPS; a++)
				if (a < step) footer += "#";
						 else footer += " ";
			g.setColor(Color.black);
			g.drawString(footer, getInsets().left + 6, getInsets().top + 23*10 + 1);
			updateToken(g);
		}
	}


	public void updateToken(Graphics g)
	{
		for (int a=0; a<44; a++)
		{
			switch (token.read(a))
			{
				case  0: g.setColor(Color.blue);	// none
						 break;
				case -1: g.setColor(Color.pink);	// white marked
						 break;
				case  1: g.setColor(Color.white);	// white
						 break;
				case -2: g.setColor(Color.red);		// black marked
						 break;
				case  2: g.setColor(Color.black);	// black
			}
			g.drawString("#", getInsets().left + 6*token.x(a) + 6, 31 + getInsets().top + 10*token.y(a));
			g.drawString("O", getInsets().left + 6*token.x(a) + 6, 31 + getInsets().top + 10*token.y(a));
			g.drawString("O", getInsets().left + 6*token.x(a) + 7, 31 + getInsets().top + 10*token.y(a));
		}
	}


	public String int2str6(int i)
	{
		char[] chars = new char[6];

		chars[0] = (char)((i/100000)%10 + '0');
		chars[1] = (char)((i/10000)%10 + '0');
		chars[2] = (char)((i/1000)%10 + '0');
		chars[3] = (char)((i/100)%10 + '0');
		chars[4] = (char)((i/10)%10 + '0');
		chars[5] = (char)((i/1)%10 + '0');
		return new String(chars);
	}


	public String gridHost(String str, int i)
	{
		str = str.replaceAll("JAVA", "/usr/bin/java");
		str = str.replaceAll("HOST", hosts[i%HOSTS]);

		return str;
	}
}
